import random


class SingletonMeta(type):
    _instances = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            instance = super().__call__(*args, **kwargs)
            cls._instances[cls] = instance
        return cls._instances[cls]


#  Pourquoi simplement changer les lettres de place quand on peut encapsuler la logique dans une classe de stratégie ?
class Strategy:
    def execute(self, data):
        pass


# Parce que la réutilisation directe des objets, c'est pour les amateurs.
# Ici, on transforme chaque syllabe avec la délicatesse d'un artisan
class Adapter:
    def adapt(self, data):
        raise NotImplementedError()


# Parce que traiter les syllabes comme de simples chaînes de caractères, c'est bon pour les "code monkeys"
class BasicSyllable:
    def get_content(self):
        raise NotImplementedError()


class TwistStrategy(Strategy):
    def execute(self, data):
        index = MonsterNameGeneratorConfig().syllables.index(data)
        return MonsterNameGeneratorConfig().syllables[index - 1]


class SyllableAdapter(Adapter):
    def adapt(self, syllable):
        if isinstance(syllable, Syllable):
            return Syllable(syllable.turn())
        raise FileNotFoundError("Message d'erreur descriptif")


class IntermediateSyllable(BasicSyllable):
    def get_content(self):
        return super().get_content()


# On aurait pu croire que la Syllabe serait une classe de base, mais non
class Syllable(IntermediateSyllable):
    def __init__(self, content):
        self.content = content

    def get_content(self):
        return self.content

    def twist(self):
        strategy = TwistStrategy()
        return strategy.execute(self)

    def turn(self):
        letters = list(self.content)
        return ''.join([letters[i] for i in [1, 0]])

    def __eq__(self, other):
        if isinstance(other, Syllable):
            return self.content == other.content
        return False


# Parce que l'univers ne peut tolérer plus d'une instance de configuration de création de noms de monstres
class MonsterNameGeneratorConfig(metaclass=SingletonMeta):
    def __init__(self):
        self.syllables = [Syllable(s) for s in ["dra", "go", "ni", "ra", "zu", "ma", "lu", "ki", "pa", "to"]]
        self._cache = {}

    @classmethod
    def create_syllable_list(cls):
        return cls().syllables


# On peut difficilement faire plus simple
class BasicMonsterNameGenerator:
    def __init__(self):
        self.config = MonsterNameGeneratorConfig.create_syllable_list()
        self._intermediate = []

    def register_syllable(self, syl):
        if syl not in self._intermediate:
            self._intermediate.append(syl)

    def get_syllable(self, prev_syl):
        if not prev_syl:
            syl = random.choice(self.config)
            self._intermediate.append(syl)
            return syl
        adapter = SyllableAdapter()
        if prev_syl in self.config:
            twisted = prev_syl.twist()
            adapted_syl = adapter.adapt(twisted)
            self._intermediate.append(adapted_syl)
            return adapted_syl
        elif self._intermediate:
            return adapter.adapt(self._intermediate[-1])
        else:
            syl = random.choice(self.config)
            self._intermediate.append(syl)
            return syl

    def generate_name(self):
        name = ''
        syl = None
        for _ in range(random.randint(2, 4)):
            syl = self.get_syllable(syl)
            name += syl.get_content()
        return name


if __name__ == "__main__":
    monster_name_generator = BasicMonsterNameGenerator()
    print(monster_name_generator.generate_name())
