import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
import gradio as gr
from sklearn.linear_model import LogisticRegression
from sklearn.model_selection import train_test_split
from sklearn.metrics import confusion_matrix, accuracy_score
from sklearn.preprocessing import StandardScaler
import statsmodels.api as sm #pour la p-value mais pas utilisé pour le moment
from sklearn.metrics import roc_curve, roc_auc_score


# === Chargement des données ===
avis = pd.read_csv("olist_order_reviews_dataset.csv")
commandes = pd.read_csv("olist_orders_dataset.csv", parse_dates=[
    "order_purchase_timestamp",
    "order_delivered_customer_date",
    "order_estimated_delivery_date"
])
articles = pd.read_csv("olist_order_items_dataset.csv")

# === Fusion pour enrichissement ===
avis = avis[avis['review_score'].notna()]
avis["avis_negatif"] = (avis["review_score"] <= 2).astype(int)

fusion = pd.merge(avis, commandes, on="order_id", how="inner")
articles_agg = articles.groupby("order_id").agg(
    prix_total=('price', 'sum'),
    nb_articles=('order_item_id', 'count')
).reset_index()
df = pd.merge(fusion, articles_agg, on="order_id", how="left")

df["delai_attente_jours"] = (df["order_delivered_customer_date"] - df["order_purchase_timestamp"]).dt.days
df["retard_livraison_jours"] = (df["order_delivered_customer_date"] - df["order_estimated_delivery_date"]).dt.days

modele_df = df[["avis_negatif", "delai_attente_jours", "retard_livraison_jours", "prix_total", "nb_articles"]].dropna()
X = modele_df.drop("avis_negatif", axis=1)
y = modele_df["avis_negatif"]
X_train, X_test, y_train, y_test = train_test_split(X, y, stratify=y, test_size=0.3, random_state=42)
modele = LogisticRegression(max_iter=1000)
modele.fit(X_train, y_train)

# === Statistiques globales ===
def stats_generales():
    total_commandes = commandes["order_id"].nunique()
    total_avis = avis["order_id"].nunique()
    avis_neg = avis[avis["review_score"] <= 2]["order_id"].nunique()
    avis_pos = avis[avis["review_score"] >= 4]["order_id"].nunique()
    avis_neutres = total_avis - avis_neg - avis_pos
    aucun_avis = total_commandes - total_avis

    return f"""
✨ STATISTIQUES GLOBALES
- Commandes : {total_commandes}
- Avis : {total_avis} dont
  - Positifs (>=4) : {avis_pos}
  - Neutres (=3) : {avis_neutres}
  - Négatifs (<=2) : {avis_neg}
- Commandes sans avis : {aucun_avis}
"""

# === Corrélation ===
def generer_corrplot():
    df_corr = modele_df
    corr_matrix = df_corr.corr()
    fig, ax = plt.subplots(figsize=(6, 4))
    sns.heatmap(corr_matrix, annot=True, cmap="coolwarm", fmt=".2f", square=True, ax=ax)
    ax.set_title("Matrice de corrélation des variables")
    return fig

# === Visualisations supplémentaires ===
def graphiques_supplementaires():
    figs = []

    fig1, ax1 = plt.subplots()
    sns.countplot(x="review_score", data=avis, ax=ax1)
    ax1.set_title("Distribution des notes")
    figs.append(fig1)

    fig2, ax2 = plt.subplots()
    sns.histplot(df["delai_attente_jours"], bins=30, kde=True, ax=ax2)
    ax2.set_title("Distribution du délai d'attente")
    figs.append(fig2)

    fig3, ax3 = plt.subplots()
    sns.histplot(df["prix_total"], bins=30, kde=True, ax=ax3)
    ax3.set_title("Distribution du prix total")
    figs.append(fig3)

    return figs



# === Normalisation des données ===
scaler = StandardScaler()
X_train_scaled = scaler.fit_transform(X_train)  # Standardiser les données d'entraînement
X_test_scaled = scaler.transform(X_test)       # Standardiser les données de test

# === Entraînement du modèle avec les données normalisées ===
modele = LogisticRegression(max_iter=1000)
modele.fit(X_train_scaled, y_train)

# === Coefficients du modèle ===
def afficher_coefficients():
    coeffs = pd.Series(modele.coef_[0], index=X.columns).sort_values(key=abs, ascending=False)
    result = "\n".join([f"{var} : {coef:.4f}" for var, coef in coeffs.items()])
    return f"📉 Coefficients du modèle logistique (normalisés) :\n\n{result}"

# === P-values des variables ===

# def afficher_p_values():
#     X_train_const = sm.add_constant(X_train_scaled)  # Ajouter une constante pour l'interception
#     logit_model = sm.Logit(y_train, X_train_const)   # Ajuster le modèle logistique
#     result = logit_model.fit(disp=False)            # Ajuster sans afficher les détails
#     p_values = pd.Series(result.pvalues, index=["Intercept"] + list(X.columns))
    
#     # Interprétation
#     interpretation = "\n".join([
#         f"{var} : p-value = {pval:.4e} {'(Significatif)' if pval < 0.05 else '(Non significatif)'}"
#         for var, pval in p_values.items()
#     ])
#     return f"📊 P-values des variables avec interprétation :\n\n{interpretation}"

# === Matrice de confusion + précision ===
def afficher_matrice_confusion():
    y_pred = modele.predict(X_test_scaled)
    cm = confusion_matrix(y_test, y_pred)
    acc = accuracy_score(y_test, y_pred)
    fig, ax = plt.subplots()
    sns.heatmap(cm, annot=True, fmt="d", cmap="Blues",
                xticklabels=["Non négatif", "Négatif"],
                yticklabels=["Non négatif", "Négatif"])
    ax.set_xlabel("Prédit")
    ax.set_ylabel("Réel")
    ax.set_title(f"Matrice de confusion (Précision : {acc * 100:.2f}%)")
    return fig

# === Courbe ROC ===
def generer_courbe_roc():
    y_pred_proba = modele.predict_proba(X_test_scaled)[:, 1]  # Probabilités pour la classe positive
    fpr, tpr, thresholds = roc_curve(y_test, y_pred_proba)  # Calcul des Taux vrai Positif et Taux faux positif
    auc_score = roc_auc_score(y_test, y_pred_proba)  # Calcul de l'AUC = Area Under the Curve
    print(f"Score AUC : {auc_score:.2f}")  # Affichage du score AUC pour information

    # Génération de la courbe ROC
    fig, ax = plt.subplots()
    ax.plot(fpr, tpr, label=f"Courbe ROC (AUC = {auc_score:.2f})", color="blue")
    ax.plot([0, 1], [0, 1], linestyle="--", color="gray", label="Classifieur Aléatoire")
    ax.set_title("Courbe ROC")
    ax.set_xlabel("Taux de Faux Positifs (FPR)")
    ax.set_ylabel("Taux de Vrais Positifs (TPR)")
    ax.legend(loc="lower right")
    return fig, auc_score

# === Prédiction ===
def predire_avis(delai_attente, retard_livraison, prix_total, nb_articles):
    x_input = pd.DataFrame([{
        "delai_attente_jours": delai_attente,
        "retard_livraison_jours": retard_livraison,
        "prix_total": prix_total,
        "nb_articles": nb_articles
    }])
    proba = modele.predict_proba(x_input)[0][1]
    return f"🔹 Probabilité d’avis négatif : {round(proba * 100, 2)} %"

# === Interface Gradio ===
with gr.Blocks(title="Olist Dashboard") as app:
    with gr.Tabs():
        # Page 1 : Statistiques générales
        with gr.Tab("Statistiques"):
            gr.Markdown("""
            # 📊 Statistiques Olist
            Aperçu global des avis clients sur les commandes.
            """)
            gr.Textbox(label="Statistiques", lines=10, value=stats_generales())
            gr.Markdown("## 📊 Graphiques supplémentaires")
            figs = graphiques_supplementaires()
            gr.Plot(value=figs[0])
            gr.Markdown("*Le score des avis permet de visualiser la répartition globale des notes laissées par les clients.*")
            gr.Plot(value=figs[1])
            gr.Markdown("*Le délai d'attente peut influencer l'expérience client, un temps long pouvant générer de l'insatisfaction.*")
            gr.Plot(value=figs[2])
            gr.Markdown("*Le prix total d'une commande est un facteur sensible : plus un client paie, plus il peut être exigeant.*")
            gr.Markdown("## 📈 Matrice de corrélation des variables influentes")
            gr.Plot(value=generer_corrplot(), label="Corrélation entre variables")
            gr.Markdown("*Cette matrice met en évidence les relations linéaires entre les variables explicatives et la variable cible.*")
            gr.Markdown("## 🧠 Importance des variables dans le modèle")
            #gr.Textbox(label="P-values", lines=10, value=afficher_p_values())
            
            gr.Textbox(label="Poids des variables (coefficients)", lines=8, value=afficher_coefficients())
            gr.Markdown("*Les coefficients du modèle de régression logistique indiquent l'influence de chaque variable sur la probabilité d’un avis négatif.*")
            gr.Image(value="modele_etoile.png", label="Modèle en étoile (exemple)")

        # Page 2 : Prédiction
        with gr.Tab("Prédiction"):
            gr.Markdown("## 🔮 Prédiction d’un avis négatif")
            delai = gr.Slider(1, 100, step=1, label="Délai d'attente (jours)")
            retard = gr.Slider(-15, 30, step=1, label="Retard de livraison (jours)")
            prix = gr.Number(label="Prix total (€)")
            nb = gr.Slider(1, 20, step=1, label="Nombre d'articles")
            btn_pred = gr.Button("Prédire")
            out_pred = gr.Textbox(label="Résultat")
            btn_pred.click(fn=predire_avis, inputs=[delai, retard, prix, nb], outputs=out_pred)

        # Page 3 : Évaluation du modèle
        with gr.Tab("Évaluation"):
            gr.Markdown("## 📌 Matrice de confusion du modèle et précision")
            gr.Plot(value=afficher_matrice_confusion(), label="Matrice de confusion")

            gr.Markdown("## 📈 Courbe ROC")
            fig_roc, auc_score = generer_courbe_roc()
            gr.Plot(value=fig_roc, label="Courbe ROC")
            gr.Markdown(f"""
            ### Interprétation de la courbe ROC :
            - **AUC (Area Under Curve)** : {auc_score:.2f}
            - Une AUC proche de 1 indique un excellent modèle.
            - Une AUC proche de 0.5 indique un modèle aléatoire.
            - La courbe ROC montre le compromis entre le taux de vrais positifs (TPR) et le taux de faux positifs (FPR).
            - Plus la courbe est proche du coin supérieur gauche, meilleur est le modèle.
            """)

app.launch()
